package MusicLandscape.entities;

import MusicLandscape.Date;
import MusicLandscape.Venue;

public class Event {
    /**
     * the artist who appeared at this event the artist of an event cannot be null
     */
    private Artist artist = new Artist();

    /**
     * the number of attendees of this event.
     */
    private int attendees = 0;

    /**
     * the date on which this event takes place
     * a null date represents an unknown date
     */
    private Date date;

    /**
     * description of this event
     * default description is an empty String
     */
    private String description = "";

    /**
     * the venue at which this event takes place
     * a null venue represents an unknown venue
     */
    private Venue venue;

    /**
     * creates a default event
     * a default event has a default artist and an empty description.
     */
    public Event() {
    }

    /**
     * gets the artist of this event
     *
     * @return the artist
     */
    public Artist getArtist() {
        return artist;
    }

    /**
     * sets the artist of this event the artist of an event cannot be null
     *
     * @param artist - the artist to set
     */
    public void setArtist(Artist artist) {
        if (artist == null) {
            return;
        }
        this.artist = artist;
    }


    /**
     * gets the number of attendees of this event
     *
     * @return the attendees
     */
    public int getAttendees() {
        return attendees;
    }

    /**
     * sets the number of attendees of this event
     * the number of attendees must be a non-negative number.
     * When called with invalid arguments this event remains unchanged.
     *
     * @param attendees - the attendees to set
     */
    public void setAttendees(int attendees) {
        if (attendees < 0) {
            return;
        }
        this.attendees = attendees;
    }

    /**
     * gets the date of this event
     * this method is defensive in the sense that it returns a copy of the date
     *
     * @return the date
     */
    public Date getDate() {
        if (date == null) {
            return null;
        }
        return new Date(date);
    }

    /**
     * sets the date of this event
     * an unknown date is represented by a null date.
     * this method is defensive in the sense that this event keep a copy of the original date
     *
     * @param date - the date to set
     */
    public void setDate(Date date) {
        if (date == null) {
            this.date = null;
            return;
        }
        this.date = new Date(date);
    }

    /**
     * gets the description of this event
     *
     * @return the description
     */
    public String getDescription() {
        return description;
    }

    /**
     * sets the description of this event description can not be null
     *
     * @param description - the description to set
     */
    public void setDescription(String description) {
        if (description == null) {
            return;
        }
        this.description = description;
    }

    /**
     * gets the venue of this event
     *
     * @return the Venue
     */
    public Venue getVenue() {
        return venue;
    }

    /**
     * sets the venue of this event
     *
     * @param venue - the venue to set
     */
    public void setVenue(Venue venue) {
        this.venue = venue;
    }
}
